//
// Copyright (c) 2016 Ricoh Co., Ltd. All rights reserved.
//
// Abstract:
//    RICOH V4 Printer Driver
//
var bidi = {};

bidi.writePJL = function(printerStream, pjl) {
  var UEL = [0x1B, 0x25, 0x2D, 0x31, 0x32, 0x33, 0x34, 0x35, 0x58];
  var HEADER = [0x40, 0x50, 0x4A, 0x4C, 0x0D, 0x0A];
  var i = 0;
  var writeData = [].concat(UEL).concat(HEADER);

  for (i = 0; i < pjl.length; i++) {
    writeData.push(pjl.charCodeAt(i));
  }
  writeData = writeData.concat(UEL);

  var bytesWritten = printerStream.write(writeData);
  return bytesWritten === writeData.length
}

bidi.readPJL = function(printerStream) {
  var readBuffer = [];
  var bytesRead = 0;
  var readString = "";
  var readEnd = false;

  while (!readEnd) {
    var buffer = printerStream.read(2048);

    readBuffer = readBuffer.concat(buffer);
    bytesRead += buffer.length;

    if (buffer.length === 0) {
      readEnd = true;
    }
    else if (buffer.length < 2048) {
      if (buffer[buffer.length - 1] === 0x0C) {
        readEnd = true;
      }
    }
  }

  for (var i = 0; (bytesRead > 0) && (i < bytesRead); i++) {
    readString += String.fromCharCode(readBuffer.shift());
  }
  return readString;
}

bidi.allInclude = function(source, target) {
  if( !source || !target || target === "\r\n" ) return false;
  var commands = target.split("\r\n");
  for(var i = 0; i < commands.length; i++ ) {
    if( commands[i] && source.indexOf(commands[i]) === -1 ) {
      return false;
    }
  }
  return true;
}

bidi.getPJL = function(printerStream, request) {
  if (!this.writePJL(printerStream, request)) return "";
  var result = "";
  var response = "";
  do {
    response = this.readPJL(printerStream);
    if( response ) {
      result += response;
    }
  } while(response && !this.allInclude(result, request));
  return result;
}

bidi.trimspace = function(value) {
  return value.replace(/^\s+|\s+$/g, '');
}

bidi.queryID = function(tray) {
  var trimmedValue = this.trimspace(tray);
  return trimmedValue === "INTRAYMANUAL" ? "INTRAYM" : trimmedValue;
}

bidi.lockQueryID = function(tray) {
  var trimmedValue = this.trimspace(tray);
  return trimmedValue === "INTRAYMANUAL" ? "INTRAYMULTI" : trimmedValue;
}

bidi.getChildNodeList = function(key, pjl) {
  var re = new RegExp(key + "\\s+\\[\\d+\\s+ENUMERATED\\s*\\]\\s*\\r\\n(?:^\\s+.+\\r\\n)+", "m");
  var list = pjl.match(re);
  if (!list) return null;
  return list[0].split("\r\n").slice(1,-1);
}

bidi.getTrayQueryCommand = function(pjl) {
  var nodelist = this.getChildNodeList("IN TRAYS", pjl);
  if(!nodelist) return "";
  var result = "";
  for (var i = 0; i < nodelist.length; i++) {
    result += "@PJL INQUIRE " + this.queryID(nodelist[i]) + "SIZE\r\n";
    result += "@PJL INQUIRE " + this.queryID(nodelist[i]) + "MEDIA\r\n";
    result += "@PJL INQUIRE " + this.lockQueryID(nodelist[i]) + "\r\n";
  }
  return result;
}

bidi.getCommonInfo = function(printerStream, printerBidiSchemaResponses) {
  var queryCommonInfo = "@PJL INFO CONFIG\r\n";
  var infoResponse = this.getPJL(printerStream, queryCommonInfo);
  if (!infoResponse) return;
  var queryTray = this.getTrayQueryCommand(infoResponse);
  if (!queryTray) return;
  var trayResponse = this.getPJL(printerStream, queryTray);
  if (!trayResponse) return;
  printerBidiSchemaResponses.addString("\\Printer.Configuration.CommonInfo:data", infoResponse + trayResponse);
  printerBidiSchemaResponses.addString("\\Printer.Configuration.CommonInfo:id", Math.random().toString());
}

bidi.getGenericInfo = function(printerStream, printerBidiSchemaResponses) {
  var queryGenericInfo = "@PJL INQUIRE DATAMODE\r\n@PJL INQUIRE STAPLE\r\n";
  var genericinfoResponse = this.getPJL(printerStream, queryGenericInfo);
  if(!genericinfoResponse) return;
  printerBidiSchemaResponses.addString("\\Printer.Configuration.GenericInfo:data", genericinfoResponse);
  printerBidiSchemaResponses.addString("\\Printer.Configuration.GenericInfo:id", Math.random().toString());
}

bidi.getGenericFunctionInfo = function(printerStream, printerBidiSchemaResponses) {
  var queryGenericFunctionInfo = "@PJL INFO GENERICFUNCKEY\r\n";
  var genericFuncResponse = this.getPJL(printerStream, queryGenericFunctionInfo);
  if(!genericFuncResponse) return;
  printerBidiSchemaResponses.addString("\\Printer.Configuration.GenericFunctionInfo:data", genericFuncResponse);
  printerBidiSchemaResponses.addString("\\Printer.Configuration.GenericFunctionInfo:id", Math.random().toString());
}

function getSchemas(scriptContext, printerStream, schemaRequests, printerBidiSchemaResponses) {
  debugger;
  var index = 0;
  for (index = 0; index < schemaRequests.length; index++) {
    if (schemaRequests[index] === "\\Printer.Cache:data") return 0;
  }
  for (index = 0; index < schemaRequests.length; index++) {
    var key = schemaRequests[index];
    switch (key) {
      case "CommonInfo":
        bidi.getCommonInfo(printerStream, printerBidiSchemaResponses);
        break;
      case "GenericInfo":
        bidi.getGenericInfo(printerStream, printerBidiSchemaResponses);
        break;
      case "GenericFunctionInfo":
        bidi.getGenericFunctionInfo(printerStream, printerBidiSchemaResponses);
        break;
      default:
    }
  }
  return 0;
}
