#pragma once

#include "Common/json/json.hpp"
#include "Common/FSecure/WinHttp/WebProxy.h"
#include "Common/FSecure/WinHttp/Constants.h"

using json = nlohmann::json; //for easy parsing of json API: https://github.com/nlohmann/json

namespace FSecure
{
	class Slack
	{
	public:

		/// Constructor for the Slack Api class.
		/// @param token - the token generated by Slack when an "app" was installed to a workspace
		/// @param proxyString - the proxy to use
		Slack(std::string const& token, std::string const& channelName);

		/// Default constructor.
		Slack() = default;

		/// Write a message to the channel this Slack object is set to.
		/// @param text - the text of the message
		/// @return - a timestamp of the message that was written to the channel.
		std::string WriteMessage(std::string const& text);

		/// Set the channel that this object uses for communications
		/// @param channel - the channelId (not name), for example CGPMGFGSH.
		void SetChannel(std::string const& channelId);

		/// set the token for this object.
		/// @param token - the textual api token.
		void SetToken(std::string const& token);

		/// Creates a channel on slack, if the channel exists already, will call ListChannels internally to get the channelId.
		/// @param channelName - the actual name of the channel, such as "general".
		/// @return - the channelId of the new or already existing channel.
		std::string CreateChannel(std::string const& channelName);

		/// Read the replies to a message
		/// @param timestamp - the timestamp of the original message, from which we can gather the replies.
		/// @return - an array of pairs containing the reply timestamp and reply text
		std::vector<std::pair<std::string, std::string>> ReadReplies(std::string const& timestamp);

		/// List all the channels in the workspace the object's token is tied to.
		/// @return - a map of {channelName -> channelId}
		std::map<std::string, std::string> ListChannels();

		/// Get all of the messages by a direction. This is a C3 specific method, used by a server relay to get client messages and vice versa.
		/// @param direction - the direction to search for (eg. "S2C").
		/// @return - a vector of timestamps, where timestamp allows replies to be read later
		std::vector<std::string> GetMessagesByDirection(std::string const& direction);

		/// Edit a previously sent message.
		/// @param message - the message to update to, this will overwrite the previous message.
		/// @param timestamp - the timestamp of the message to update.
		void UpdateMessage(std::string const& message, std::string const& timestamp);

		/// Create a thread on a message by writing a reply to it.
		/// @param text - the text to send as a reply.
		/// @param timestamp - the timestamp of the message that the reply is for.
		void WriteReply(std::string const& text, std::string const& timestamp);

		/// Use Slack's file API to upload data as files. This is useful when a payload is large (for example during implant staging).
		/// This function is called internally whenever a WriteReply is called with a payload of more than 120k characters.
		/// @param data - the data to be sent.
		/// @param ts - the timestamp, needed as this method is only used during WriteReply.
		void UploadFile(ByteView data, std::string const& ts);

		/// Delete a message from the channel
		/// @param timestamp - the timestamp of the message to delete.
		void DeleteMessage(std::string const& timestamp);

	private:

		/// The channel through which messages are sent and received, will be sent when the object is created.
		std::string m_Channel;

		/// The Slack API token that allows the object access to the workspace. Needs to be manually created as described in documentation.
		std::string m_Token;

		/// Hold proxy settings
		WinHttp::WebProxy m_ProxyConfig;

		/// Send http request, uses preset token for authentication
		ByteVector SendHttpRequest(std::string const& host, std::optional<WinHttp::ContentType> contentType = {}, std::string const& data = "");

		/// Send http request with json data, uses preset token for authentication
		json SendJsonRequest(std::string const& url, json const& data);

		/// Send http GET request, uses preset token for authentication, expect response of application/json type
		json GetJsonResponse(std::string const& url);

		/// Use Slack's File API to retrieve files.
		/// @param url - the url where the file can be retrieved.
		/// @return - the data within the file.
		std::string GetFile(std::string const& url);

	};

}
